<?php
/**
 * Environment
 *
 * Copyright (c) 2014, Sebastian Bergmann <sebastian@phpunit.de>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *
 *   * Neither the name of Sebastian Bergmann nor the names of his
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @package    Environment
 * @author     Sebastian Bergmann <sebastian@phpunit.de>
 * @copyright  2014 Sebastian Bergmann <sebastian@phpunit.de>
 * @license    http://www.opensource.org/licenses/BSD-3-Clause  The BSD 3-Clause License
 * @link       http://www.github.com/sebastianbergmann/environment
 */

namespace SebastianBergmann\Environment;

/**
 * Utility class for HHVM/PHP environment handling.
 *
 * @package    Environment
 * @author     Sebastian Bergmann <sebastian@phpunit.de>
 * @copyright  2014 Sebastian Bergmann <sebastian@phpunit.de>
 * @license    http://www.opensource.org/licenses/BSD-3-Clause  The BSD 3-Clause License
 * @link       http://www.github.com/sebastianbergmann/environment
 */
class Runtime {
	/**
	 * @var string
	 */
	private static $binary;

	/**
	 * Returns true when the runtime used is HHVM or
	 * the runtime used is PHP + Xdebug.
	 *
	 * @return boolean
	 */
	public function canCollectCodeCoverage () {
		return $this->isHHVM() || $this->hasXdebug();
	}

	/**
	 * Returns the path to the binary of the current runtime.
	 * Appends ' --php' to the path when the runtime is HHVM.
	 *
	 * @return string
	 */
	public function getBinary () {
		// HHVM
		if ( self::$binary === null && $this->isHHVM() ) {
			if ( ( self::$binary = getenv( 'PHP_BINARY' ) ) === false ) {
				self::$binary = PHP_BINARY;
			}

			self::$binary = escapeshellarg( self::$binary ) . ' --php';
		}

		// PHP >= 5.4.0
		if ( self::$binary === null && defined( 'PHP_BINARY' ) ) {
			self::$binary = escapeshellarg( PHP_BINARY );
		}

		// PHP < 5.4.0
		if ( self::$binary === null ) {
			if ( PHP_SAPI == 'cli' && isset( $_SERVER['_'] ) ) {
				if ( strpos( $_SERVER['_'], 'phpunit' ) !== false ) {
					$file = file( $_SERVER['_'] );

					if ( strpos( $file[0], ' ' ) !== false ) {
						$tmp          = explode( ' ', $file[0] );
						self::$binary = escapeshellarg( trim( $tmp[1] ) );
					} else {
						self::$binary = escapeshellarg( ltrim( trim( $file[0] ), '#!' ) );
					}
				} elseif ( strpos( basename( $_SERVER['_'] ), 'php' ) !== false ) {
					self::$binary = escapeshellarg( $_SERVER['_'] );
				}
			}
		}

		if ( self::$binary === null ) {
			$possibleBinaryLocations = array (
				PHP_BINDIR . '/php',
				PHP_BINDIR . '/php-cli.exe',
				PHP_BINDIR . '/php.exe'
			);

			foreach ( $possibleBinaryLocations as $binary ) {
				if ( is_readable( $binary ) ) {
					self::$binary = escapeshellarg( $binary );
					break;
				}
			}
		}

		if ( self::$binary === null ) {
			self::$binary = 'php';
		}

		return self::$binary;
	}

	/**
	 * @return string
	 */
	public function getNameWithVersion () {
		return $this->getName() . ' ' . $this->getVersion();
	}

	/**
	 * @return string
	 */
	public function getName () {
		if ( $this->isHHVM() ) {
			return 'HHVM';
		} else {
			return 'PHP';
		}
	}

	/**
	 * @return string
	 */
	public function getVendorUrl () {
		if ( $this->isHHVM() ) {
			return 'http://hhvm.com/';
		} else {
			return 'http://php.net/';
		}
	}

	/**
	 * @return string
	 */
	public function getVersion () {
		if ( $this->isHHVM() ) {
			return HHVM_VERSION;
		} else {
			return PHP_VERSION;
		}
	}

	/**
	 * Returns true when the runtime used is PHP and Xdebug is loaded.
	 *
	 * @return boolean
	 */
	public function hasXdebug () {
		return $this->isPHP() && extension_loaded( 'xdebug' );
	}

	/**
	 * Returns true when the runtime used is HHVM.
	 *
	 * @return boolean
	 */
	public function isHHVM () {
		return defined( 'HHVM_VERSION' );
	}

	/**
	 * Returns true when the runtime used is PHP.
	 *
	 * @return boolean
	 */
	public function isPHP () {
		return ! $this->isHHVM();
	}
}
